import os
import jax
os.environ["KERAS_BACKEND"] = "jax"

# use gpu 0
os.environ["CUDA_VISIBLE_DEVICES"] = "0"

import pathlib
import random
import string
import re
import tensorflow.data as tf_data
import tensorflow.strings as tf_strings
import keras
from keras import layers
from src.standard_attention import StandardMultiHeadAttention
from src.optimised_attention import OptimisedAttention
from src.efficient_attention import EfficientAttention
from keras.layers import TextVectorization
from keras.callbacks import LearningRateScheduler
from nltk.translate.bleu_score import corpus_bleu
import pandas as pd
import time
import math

text_file = keras.utils.get_file(
    fname="spa-eng.zip",
    origin="http://storage.googleapis.com/download.tensorflow.org/data/spa-eng.zip",
    extract=True,
)

text_file = pathlib.Path(text_file).parent / "spa-eng" / "spa.txt"

with open(text_file) as f:
    lines = f.read().split("\n")[:-1]
text_pairs = []
for line in lines:
    eng, spa = line.split("\t")
    spa = "[start] " + spa + " [end]"
    text_pairs.append((eng, spa))

# download the spanish-english translation file from europarl
text_file_europarl = keras.utils.get_file(
    fname="es-en.tgz",
    origin="https://www.statmt.org/europarl/v7/es-en.tgz",
    extract=True,
)

text_file_en = pathlib.Path(text_file_europarl).parent / "europarl-v7.es-en.en"

with open(text_file_en) as f:
    eng_lines = f.read().split("\n")[:-1]

# now open the spanish file
text_file_es = pathlib.Path(text_file_europarl).parent / "europarl-v7.es-en.es"

with open(text_file_es) as f:
    esp_lines = f.read().split("\n")[:-1]

# append the english and spanish lines to the text_pairs list
for eng, esp in zip(eng_lines, esp_lines):
    esp = "[start] " + esp + " [end]"
    text_pairs.append((eng, esp))

# for _ in range(5):
#     print(random.choice(text_pairs))



#set the seed for reproducibility
random.seed(1)
random.shuffle(text_pairs)
num_val_samples = int(0.001 * len(text_pairs))
num_train_samples = len(text_pairs) - 2 * num_val_samples
train_pairs = text_pairs[:num_train_samples]
val_pairs = text_pairs[num_train_samples : num_train_samples + num_val_samples]
test_pairs = text_pairs[num_train_samples + num_val_samples :]

print(f"{len(text_pairs)} total pairs")
print(f"{len(train_pairs)} training pairs")
print(f"{len(val_pairs)} validation pairs")
print(f"{len(test_pairs)} test pairs")


strip_chars = string.punctuation + "¿"
strip_chars = strip_chars.replace("[", "")
strip_chars = strip_chars.replace("]", "")

vocab_size = 25000
sequence_length = 40
batch_size = 768

def custom_standardization(input_string):
    lowercase = tf_strings.lower(input_string)
    return tf_strings.regex_replace(lowercase, "[%s]" % re.escape(strip_chars), "")


eng_vectorization = TextVectorization(
    max_tokens=vocab_size,
    output_mode="int",
    output_sequence_length=sequence_length,
)
spa_vectorization = TextVectorization(
    max_tokens=vocab_size,
    output_mode="int",
    output_sequence_length=sequence_length + 1,
    standardize=custom_standardization,
)
train_eng_texts = [pair[0] for pair in train_pairs]
train_spa_texts = [pair[1] for pair in train_pairs]
eng_vectorization.adapt(train_eng_texts)
spa_vectorization.adapt(train_spa_texts)


def format_dataset(eng, spa):
    eng = eng_vectorization(eng)
    spa = spa_vectorization(spa)
    return (
        {
            "encoder_inputs": eng,
            "decoder_inputs": spa[:, :-1],
        },
        spa[:, 1:],
    )


def make_dataset(pairs):
    eng_texts, spa_texts = zip(*pairs)
    eng_texts = list(eng_texts)
    spa_texts = list(spa_texts)
    dataset = tf_data.Dataset.from_tensor_slices((eng_texts, spa_texts))
    dataset = dataset.batch(batch_size)
    dataset = dataset.map(format_dataset)
    return dataset.cache().shuffle(2048).prefetch(16)


train_ds = make_dataset(train_pairs)
val_ds = make_dataset(val_pairs)
test_ds = make_dataset(test_pairs)

for inputs, targets in train_ds.take(1):
    print(f'inputs["encoder_inputs"].shape: {inputs["encoder_inputs"].shape}')
    print(f'inputs["decoder_inputs"].shape: {inputs["decoder_inputs"].shape}')
    print(f"targets.shape: {targets.shape}")


import keras.ops as ops


class TransformerEncoder(layers.Layer):
    def __init__(self, ATTENTION_ARCH, proj_dim, dense_dim, num_heads, **kwargs):
        super().__init__(**kwargs)
        self.proj_dim = proj_dim
        self.dense_dim = dense_dim
        self.num_heads = num_heads
        self.attention = ATTENTION_ARCH(
            num_heads=num_heads, key_dim=(proj_dim//num_heads)
        )
        self.dense_proj = keras.Sequential(
            [
                layers.Dense(dense_dim, activation="gelu"),
                layers.Dense(proj_dim),
            ]
        )
        self.layernorm_1 = layers.LayerNormalization()
        self.layernorm_2 = layers.LayerNormalization()
        self.supports_masking = True

    def call(self, inputs, mask=None):
        if mask is not None:
            padding_mask = ops.cast(mask[:, None, :], dtype="int32")
        else:
            padding_mask = None

        attention_output = self.attention(
            query=inputs, value=inputs, key=inputs, attention_mask=padding_mask
        )
        proj_input = self.layernorm_1(inputs + attention_output)
        proj_output = self.dense_proj(proj_input)
        return self.layernorm_2(proj_input + proj_output)

    def get_config(self):
        config = super().get_config()
        config.update(
            {
                "proj_dim": self.proj_dim,
                "dense_dim": self.dense_dim,
                "num_heads": self.num_heads,
            }
        )
        return config


class PositionalEmbedding(layers.Layer):
    def __init__(self, sequence_length, vocab_size, embed_dim, **kwargs):
        super().__init__(**kwargs)
        self.token_embeddings = layers.Embedding(
            input_dim=vocab_size, output_dim=embed_dim
        )
        self.position_embeddings = layers.Embedding(
            input_dim=sequence_length, output_dim=embed_dim
        )
        self.sequence_length = sequence_length
        self.vocab_size = vocab_size
        self.embed_dim = embed_dim

    def call(self, inputs):
        length = ops.shape(inputs)[-1]
        positions = ops.arange(0, length, 1)
        embedded_tokens = self.token_embeddings(inputs)
        embedded_positions = self.position_embeddings(positions)
        return embedded_tokens + embedded_positions

    def compute_mask(self, inputs, mask=None):
        if mask is None:
            return None
        else:
            return ops.not_equal(inputs, 0)

    def get_config(self):
        config = super().get_config()
        config.update(
            {
                "sequence_length": self.sequence_length,
                "vocab_size": self.vocab_size,
                "embed_dim": self.embed_dim,
            }
        )
        return config


class TransformerDecoder(layers.Layer):
    def __init__(self, ATTENTION_ARCH, proj_dim, latent_dim, num_heads, **kwargs):
        super().__init__(**kwargs)
        self.proj_dim = proj_dim
        self.latent_dim = latent_dim
        self.num_heads = num_heads
        self.attention_1 = ATTENTION_ARCH(
            num_heads=num_heads, key_dim=(proj_dim//num_heads)
        )
        self.attention_2 = ATTENTION_ARCH(
            num_heads=num_heads, key_dim=(proj_dim//num_heads)
        )
        self.dense_proj = keras.Sequential(
            [
                layers.Dense(latent_dim, activation="gelu"),
                layers.Dense(proj_dim),
            ]
        )
        self.layernorm_1 = layers.LayerNormalization()
        self.layernorm_2 = layers.LayerNormalization()
        self.layernorm_3 = layers.LayerNormalization()
        self.supports_masking = True

    def call(self, inputs, encoder_outputs, mask=None):
        causal_mask = self.get_causal_attention_mask(inputs)
        if mask is not None:
            padding_mask = ops.cast(mask[:, None, :], dtype="int32")
            padding_mask = ops.minimum(padding_mask, causal_mask)
        else:
            padding_mask = None

        attention_output_1 = self.attention_1(
            query=inputs, value=inputs, key=inputs, attention_mask=causal_mask
        )
        out_1 = self.layernorm_1(inputs + attention_output_1)

        attention_output_2 = self.attention_2(
            query=out_1,
            value=encoder_outputs,
            key=encoder_outputs,
            attention_mask=padding_mask,
        )
        out_2 = self.layernorm_2(out_1 + attention_output_2)

        proj_output = self.dense_proj(out_2)
        return self.layernorm_3(out_2 + proj_output)

    def get_causal_attention_mask(self, inputs):
        input_shape = ops.shape(inputs)
        batch_size, sequence_length = input_shape[0], input_shape[1]

        with jax.ensure_compile_time_eval():
            i = ops.arange(sequence_length)[:, None]
            j = ops.arange(sequence_length)
            mask = ops.cast(i >= j, dtype="int32")
            mask = ops.reshape(mask, (1, input_shape[1], input_shape[1]))
            mult = ops.concatenate(
                [ops.expand_dims(batch_size, -1), ops.convert_to_tensor([1, 1])],
                axis=0,
            )
            return ops.tile(mask, mult)

    def get_config(self):
        config = super().get_config()
        config.update(
            {
                "proj_dim": self.proj_dim,
                "latent_dim": self.latent_dim,
                "num_heads": self.num_heads,
            }
        )
        return config


class Translator:
    def __init__(self, transformer):
        self.transformer = transformer
        self.spa_vocab = spa_vectorization.get_vocabulary()
        self.spa_index_lookup = dict(zip(range(len(self.spa_vocab)), self.spa_vocab))
        self.max_decoded_sentence_length = 40

    def decode_sequence(self, input_sentence):
        tokenized_input_sentence = eng_vectorization([input_sentence])
        decoded_sentence = "[start]"

        for i in range(self.max_decoded_sentence_length):
            tokenized_target_sentence = spa_vectorization([decoded_sentence])[:, :-1]
            predictions = self.transformer([tokenized_input_sentence, tokenized_target_sentence])

            sampled_token_index = ops.convert_to_numpy(
                ops.argmax(predictions[0, i, :])
            ).item(0)
            sampled_token = self.spa_index_lookup[sampled_token_index]
            decoded_sentence += " " + sampled_token

            if sampled_token == "[end]":
                break

        return decoded_sentence


# create a time callback class to measure the time per epoch
class TimeHistory(keras.callbacks.Callback):
    def on_train_begin(self, logs={}):
        self.times = []

    def on_epoch_begin(self, epoch, logs={}):
        self.epoch_start_time = time.time()

    def on_epoch_end(self, epoch, logs={}):
        self.times.append(time.time() - self.epoch_start_time)

def create_encoder(ATTN_ARCH, sequence_length, vocab_size, latent_dim, dense_dim, num_heads, num_encoder_blocks, dropout_rate):
    encoder_inputs = keras.Input(shape=(None,), dtype="int64", name="encoder_inputs")
    x = PositionalEmbedding(sequence_length=sequence_length,
                            vocab_size=vocab_size,
                            embed_dim=latent_dim)(encoder_inputs)
    x = layers.Dropout(dropout_rate)(x)  # Dropout after positional embedding

    # Add multiple TransformerEncoder blocks
    encoder_outputs = x
    for _ in range(num_encoder_blocks):
        encoder_outputs = TransformerEncoder(ATTENTION_ARCH=ATTN_ARCH,
                                             proj_dim=latent_dim,
                                             dense_dim=dense_dim,
                                             num_heads=num_heads,
                                             )(encoder_outputs)
        encoder_outputs = layers.Dropout(dropout_rate)(encoder_outputs)  # Dropout between TransformerEncoder blocks

    encoder = keras.Model(encoder_inputs, encoder_outputs)
    return encoder, encoder_inputs, encoder_outputs


def create_decoder(ATTN_ARCH, sequence_length, vocab_size, latent_dim, dense_dim, num_heads, num_decoder_blocks, dropout_rate):
    decoder_inputs = keras.Input(shape=(None,), dtype="int64", name="decoder_inputs")
    encoded_seq_inputs = keras.Input(shape=(None, latent_dim), name="decoder_state_inputs")
    x = PositionalEmbedding(sequence_length=sequence_length,
                            vocab_size=vocab_size,
                            embed_dim=latent_dim)(decoder_inputs)
    x = layers.Dropout(dropout_rate)(x)  # Dropout after positional embedding

    # Add multiple TransformerDecoder blocks
    decoder_outputs = x
    for _ in range(num_decoder_blocks):
        decoder_outputs = TransformerDecoder(ATTENTION_ARCH=ATTN_ARCH,
                                             proj_dim=latent_dim,
                                             latent_dim=latent_dim,
                                             num_heads=num_heads,
                                             )(decoder_outputs, encoded_seq_inputs)
        decoder_outputs = layers.Dropout(dropout_rate)(decoder_outputs)  # Dropout between TransformerDecoder blocks

    decoder_outputs = layers.Dropout(dropout_rate)(decoder_outputs)  # Dropout before final Dense layer
    decoder_outputs = layers.Dense(vocab_size, activation="softmax")(decoder_outputs)
    decoder = keras.Model([decoder_inputs, encoded_seq_inputs], decoder_outputs)
    return decoder, decoder_inputs


def calculate_bleu_score(input_pairs, translator):
    actual, predicted = [], []

    for eng_inp, spa_inp in input_pairs:
        actual.append(spa_inp.split())
        predicted.append(translator.decode_sequence(eng_inp).split())

    blue_score = corpus_bleu([[a] for a in actual], predicted)

    return round(100 * blue_score, 3)


# Defining the parameters
latent_dim = 1024
dense_dim = latent_dim
epochs = 6  # This should be at least 30 for convergence
NUM_BLEU_SAMPLES = 200
ATTN_ARCHS = [EfficientAttention, StandardMultiHeadAttention, OptimisedAttention]
NUM_OF_HEADS = [4, 2, 1]
num_encoder_blocks = 1
num_decoder_blocks = 2
dropout_rate = 0.15

for attn_arch in ATTN_ARCHS:
    for num_heads in NUM_OF_HEADS:

        print(f"Training model with {attn_arch.__name__} and {num_heads} heads")

        initial_lr = 0.00075
        multiply_factor = 0.25

        # define the learning rate scheduler
        def lr_scheduler(epoch):
            return initial_lr * (multiply_factor ** epoch)

        encoder, encoder_inputs, encoder_outputs = create_encoder(attn_arch, sequence_length, vocab_size,
                                                                  latent_dim, dense_dim, num_heads,
                                                                  num_encoder_blocks, dropout_rate)
        decoder, decoder_inputs = create_decoder(attn_arch, sequence_length, vocab_size,
                                                 latent_dim, dense_dim, num_heads,
                                                 num_decoder_blocks, dropout_rate)

        decoder_outputs = decoder([decoder_inputs, encoder_outputs])
        transformer = keras.Model(
            [encoder_inputs, decoder_inputs], decoder_outputs, name="transformer"
        )

        optimizer = keras.optimizers.AdamW(learning_rate=initial_lr,)

        transformer.summary()
        transformer.compile(
            optimizer=optimizer, loss="sparse_categorical_crossentropy", metrics=["accuracy"]
        )

        time_callback = TimeHistory()

        history = transformer.fit(
            train_ds, validation_data=val_ds,
            epochs=epochs,
            callbacks=[LearningRateScheduler(lr_scheduler), time_callback])

        test_loss, test_acc = transformer.evaluate(test_ds)

        # Create an instance of the Translator class
        translator = Translator(transformer)

        test_eng_texts = [pair[0] for pair in test_pairs]

        # calculate BLEU score on test set
        random_test_pairs = random.sample(test_pairs, NUM_BLEU_SAMPLES)
        print("Calculating BLEU score on test set...")
        test_blue_score = calculate_bleu_score(random_test_pairs, translator)

        # calculate BLEU score on validation set
        random_val_pairs = random.sample(val_pairs, NUM_BLEU_SAMPLES)
        print("Calculating BLEU score on validation set...")
        val_blue_score = calculate_bleu_score(random_val_pairs, translator)

        # calculate BLEU score on train set
        random_train_pairs = random.sample(train_pairs, NUM_BLEU_SAMPLES)
        print("Calculating BLEU score on train set...")
        train_blue_score = calculate_bleu_score(random_train_pairs, translator)

        save_dir = "./results/translation/" + attn_arch.__name__ + f"_{num_heads}_heads"
        history_file_path = save_dir + "/history.csv"

        os.makedirs(os.path.dirname(history_file_path), exist_ok=True)

        history_df = pd.DataFrame(history.history)
        history_df = history_df.round(4)
        history_df.to_csv(history_file_path, sep='\t', index=False)

        # save the test results with 4 decimal places and the BLEU scores
        test_results_file_path = save_dir + "/test_results.csv"

        test_results_df = pd.DataFrame([[round(test_loss, 4), round(test_acc, 4), round(test_blue_score, 2),
                                         round(val_blue_score, 2), round(train_blue_score, 2)]],
                                       columns=["loss", "accuracy", "test_bleu_score",
                                                "val_bleu_score", "train_bleu_score"])

        test_results_df.to_csv(test_results_file_path, sep='\t', index=False)

        # save general info file with number of attention parameters (transformer.layers[3].attenion.count_params()),
        # , d_m, d_k and and average epoch time (excluding the first epoch)
        general_info_file_path = save_dir + "/general_info.csv"

        num_of_attention_params = transformer.layers[3].attention.count_params()
        average_epoch_time_without_first_epoch = sum(time_callback.times[1:]) / (len(time_callback.times) - 1)
        average_epoch_time = sum(time_callback.times) / len(time_callback.times)

        general_info_df = pd.DataFrame(
            [[num_of_attention_params, latent_dim, latent_dim//num_heads, num_heads,
              round(average_epoch_time_without_first_epoch, 3), round(average_epoch_time, 3)]],
            columns=["num_of_attention_params", "d_m", "d_k", "num_heads",
                     "average_epoch_time_excluding_first_epoch", "average_epoch_time"])

        general_info_df.to_csv(general_info_file_path, sep='\t', index=False)


